/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.accumulo.classloader.ccl;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static org.apache.accumulo.classloader.ccl.CachingClassLoaderFactory.PROP_ALLOWED_URLS;
import static org.apache.accumulo.classloader.ccl.CachingClassLoaderFactory.PROP_CACHE_DIR;
import static org.apache.accumulo.classloader.ccl.CachingClassLoaderFactory.PROP_GRACE_PERIOD;
import static org.apache.accumulo.classloader.ccl.CachingClassLoaderFactoryTest.DESC;
import static org.apache.accumulo.classloader.ccl.CachingClassLoaderFactoryTest.MONITOR_INTERVAL_SECS;
import static org.apache.accumulo.classloader.ccl.LocalStore.WORKING_DIR;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertInstanceOf;
import static org.junit.jupiter.api.Assertions.assertIterableEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

import org.apache.accumulo.classloader.ccl.manifest.Manifest;
import org.apache.accumulo.classloader.ccl.manifest.Resource;
import org.apache.accumulo.core.client.Accumulo;
import org.apache.accumulo.core.client.AccumuloClient;
import org.apache.accumulo.core.client.AccumuloException;
import org.apache.accumulo.core.client.AccumuloSecurityException;
import org.apache.accumulo.core.client.IteratorSetting;
import org.apache.accumulo.core.client.Scanner;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.clientImpl.AccumuloServerException;
import org.apache.accumulo.core.clientImpl.ClientContext;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.iterators.IteratorUtil.IteratorScope;
import org.apache.accumulo.core.metadata.schema.Ample;
import org.apache.accumulo.core.metadata.schema.TabletMetadata;
import org.apache.accumulo.core.metadata.schema.TabletsMetadata;
import org.apache.accumulo.harness.MiniClusterConfigurationCallback;
import org.apache.accumulo.harness.SharedMiniClusterBase;
import org.apache.accumulo.miniclusterImpl.MiniAccumuloConfigImpl;
import org.apache.accumulo.test.TestIngest;
import org.apache.accumulo.test.TestIngest.IngestParams;
import org.apache.accumulo.test.VerifyIngest;
import org.apache.accumulo.test.VerifyIngest.VerifyParams;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.CleanupMode;
import org.junit.jupiter.api.io.TempDir;

public class MiniAccumuloClusterClassLoaderFactoryTest extends SharedMiniClusterBase {

  private static class TestMACConfiguration implements MiniClusterConfigurationCallback {

    @Override
    public void configureMiniCluster(MiniAccumuloConfigImpl cfg,
        org.apache.hadoop.conf.Configuration coreSite) {
      cfg.setProperty(Property.TSERV_NATIVEMAP_ENABLED.getKey(), "false");
      cfg.setProperty(Property.GENERAL_CONTEXT_CLASSLOADER_FACTORY.getKey(),
          CachingClassLoaderFactory.class.getName());
      cfg.setProperty(PROP_CACHE_DIR, baseCacheDir.toUri().toString());
      cfg.setProperty(PROP_ALLOWED_URLS, ".*");
      cfg.setProperty(PROP_GRACE_PERIOD, "1");
    }
  }

  @TempDir(cleanup = CleanupMode.ON_SUCCESS)
  private static Path tempDir;

  private static final String ITER_CLASS_NAME =
      "org.apache.accumulo.classloader.vfs.examples.ExampleIterator";

  private static URL jarAOrigLocation;
  private static URL jarBOrigLocation;
  private static Path baseCacheDir;

  @BeforeAll
  public static void beforeAll() throws Exception {
    baseCacheDir = Files.createTempDirectory(tempDir, "base-");

    // Find the Test jar files
    jarAOrigLocation = MiniAccumuloClusterClassLoaderFactoryTest.class
        .getResource("/ExampleIteratorsA/example-iterators-a.jar");
    assertNotNull(jarAOrigLocation);
    jarBOrigLocation = MiniAccumuloClusterClassLoaderFactoryTest.class
        .getResource("/ExampleIteratorsB/example-iterators-b.jar");
    assertNotNull(jarBOrigLocation);

    startMiniClusterWithConfig(new TestMACConfiguration());
  }

  @AfterAll
  public static void afterAll() throws Exception {
    stopMiniCluster();
  }

  @Test
  public void testClassLoader() throws Exception {
    final var workingDirPath = baseCacheDir.resolve(WORKING_DIR);
    final var jsonDirPath = tempDir.resolve("simulatedRemoteContextFiles");
    Files.createDirectory(jsonDirPath);

    // Create a manifest that only references jar A
    final var manifest = Manifest.create(DESC, MONITOR_INTERVAL_SECS, "SHA-256", jarAOrigLocation);
    final String manifestJson = manifest.toJson();
    final File testFile = jsonDirPath.resolve("testManifest.json").toFile();
    Files.writeString(testFile.toPath(), manifestJson);
    assertTrue(Files.exists(testFile.toPath()));

    Resource jarAResource = manifest.getResources().iterator().next();
    String jarALocalFileName = LocalStore.localResourceName(jarAResource);

    final String[] names = this.getUniqueNames(1);
    try (AccumuloClient client =
        Accumulo.newClient().from(getCluster().getClientProperties()).build()) {

      List<String> tservers = client.instanceOperations().getTabletServers();
      Collections.sort(tservers);
      assertTrue(tservers.size() >= 2, "Expected at least 2 tservers, but saw " + tservers);

      final String tableName = names[0];

      final IngestParams params = new IngestParams(client.properties(), tableName, 100);
      params.cols = 10;
      params.dataSize = 10;
      params.startRow = 0;
      params.columnFamily = "test";
      params.createTable = true;
      params.numsplits = 3;
      params.flushAfterRows = 0;

      TestIngest.createTable(client, params);

      // Confirm 4 tablets, spread across all tablet servers
      client.instanceOperations().waitForBalance();

      final List<TabletMetadata> tm = getLocations(((ClientContext) client).getAmple(),
          client.tableOperations().tableIdMap().get(tableName));
      assertEquals(4, tm.size());

      final Set<String> tabletLocations = new TreeSet<>();
      tm.forEach(t -> tabletLocations.add(t.getLocation().getHostPort()));
      assertTrue(tabletLocations.size() >= 2,
          "Expected at least 2 tablet locations, but saw " + tabletLocations);

      // both collections are sorted
      assertIterableEquals(tservers, tabletLocations);

      TestIngest.ingest(client, params);

      final VerifyParams vp = new VerifyParams(client.properties(), tableName, params.rows);
      vp.cols = params.cols;
      vp.rows = params.rows;
      vp.dataSize = params.dataSize;
      vp.startRow = params.startRow;
      vp.columnFamily = params.columnFamily;
      vp.cols = params.cols;
      VerifyIngest.verifyIngest(client, vp);

      // Set the table classloader context. The context is the URL to the manifest file
      final String contextURL = testFile.toURI().toURL().toString();
      client.tableOperations().setProperty(tableName, Property.TABLE_CLASSLOADER_CONTEXT.getKey(),
          contextURL);

      // check that the table is returning unique values
      // before applying the iterator
      final byte[] jarAValueBytes = "foo".getBytes(UTF_8);
      assertEquals(0, countExpectedValues(client, tableName, jarAValueBytes));
      Set<Path> refFiles = getReferencedFiles(workingDirPath);
      assertEquals(1, refFiles.size(), refFiles::toString);
      assertTrue(refFiles.stream().anyMatch(p -> p.endsWith(jarALocalFileName)));

      // Attach a scan iterator to the table
      IteratorSetting is = new IteratorSetting(101, "example", ITER_CLASS_NAME);
      client.tableOperations().attachIterator(tableName, is, EnumSet.of(IteratorScope.scan));

      // confirm that all values get transformed to "foo"
      // by the iterator
      int count = 0;
      while (count != 1000) {
        count = countExpectedValues(client, tableName, jarAValueBytes);
      }
      refFiles = getReferencedFiles(workingDirPath);
      assertEquals(1, refFiles.size(), refFiles::toString);
      assertTrue(refFiles.stream().anyMatch(p -> p.endsWith(jarALocalFileName)));

      // Update to point to jar B
      final Manifest update =
          Manifest.create(DESC, MONITOR_INTERVAL_SECS, "SHA-512", jarBOrigLocation);
      final String updateJson = update.toJson();
      Files.writeString(testFile.toPath(), updateJson);
      assertTrue(Files.exists(testFile.toPath()));

      Resource jarBResource = update.getResources().iterator().next();
      String jarBLocalFileName = LocalStore.localResourceName(jarBResource);

      // Wait 2x the monitor interval
      Thread.sleep(2 * MONITOR_INTERVAL_SECS * 1000);

      // Rescan with same iterator class name
      // confirm that all values get transformed to "bar"
      // by the iterator
      final byte[] jarBValueBytes = "bar".getBytes(UTF_8);
      assertEquals(1000, countExpectedValues(client, tableName, jarBValueBytes));
      refFiles = getReferencedFiles(workingDirPath);
      assertEquals(2, refFiles.size(), refFiles::toString);
      assertTrue(refFiles.stream().anyMatch(p -> p.endsWith(jarALocalFileName)));
      assertTrue(refFiles.stream().anyMatch(p -> p.endsWith(jarBLocalFileName)));

      // Copy jar A, create a manifest using the copy, then remove the copy so that it's not
      // found when the context classloader updates.
      var jarAPath = Path.of(jarAOrigLocation.toURI());
      var jarAPathParent = jarAPath.getParent();
      assertNotNull(jarAPathParent);
      var jarACopy = jarAPathParent.resolve("jarACopy.jar");
      assertTrue(!Files.exists(jarACopy));
      Files.copy(jarAPath, jarACopy, REPLACE_EXISTING);
      assertTrue(Files.exists(jarACopy));

      final var update2 =
          Manifest.create(DESC, MONITOR_INTERVAL_SECS, "SHA-512", jarACopy.toUri().toURL());
      Files.delete(jarACopy);
      assertTrue(!Files.exists(jarACopy));

      final String updateJson2 = update2.toJson();
      Files.writeString(testFile.toPath(), updateJson2);
      assertTrue(Files.exists(testFile.toPath()));

      // Wait 2x the monitor interval
      Thread.sleep(2 * MONITOR_INTERVAL_SECS * 1000);

      // Rescan and confirm that all values get transformed to "bar"
      // by the iterator. The previous classloader is still being used after
      // the monitor interval because the jar referenced does not exist.
      assertEquals(1000, countExpectedValues(client, tableName, jarBValueBytes));
      refFiles = getReferencedFiles(workingDirPath);
      assertEquals(2, refFiles.size(), refFiles::toString);
      assertTrue(refFiles.stream().anyMatch(p -> p.endsWith(jarALocalFileName)));
      assertTrue(refFiles.stream().anyMatch(p -> p.endsWith(jarBLocalFileName)));

      // Wait 2 minutes, 2 times the UPDATE_FAILURE_GRACE_PERIOD_MINS
      Thread.sleep(120_000);

      // Scan of table with iterator setting should now fail.
      final Scanner scanner2 = client.createScanner(tableName);
      var re = assertThrows(RuntimeException.class, () -> scanner2.iterator().hasNext());
      assertInstanceOf(AccumuloServerException.class, re.getCause());
    }
  }

  private Set<Path> getReferencedFiles(Path workingDirPath) throws IOException {
    // get all files in subdirectories in working directory
    try (var s = Files.walk(workingDirPath).filter(p -> p.toFile().isFile())
        .filter(p -> p.getNameCount() == workingDirPath.getNameCount() + 2)
        .map(Path::getFileName)) {
      return s.collect(Collectors.toSet());
    }
  }

  private int countExpectedValues(AccumuloClient client, String table, byte[] expectedValue)
      throws TableNotFoundException, AccumuloSecurityException, AccumuloException {
    Scanner scanner = client.createScanner(table);
    int count = 0;
    for (Entry<Key,Value> e : scanner) {
      if (Arrays.equals(e.getValue().get(), expectedValue)) {
        count++;
      }
    }
    return count;
  }

  private static List<TabletMetadata> getLocations(Ample ample, String tableId) {
    try (TabletsMetadata tabletsMetadata = ample.readTablets().forTable(TableId.of(tableId))
        .fetch(TabletMetadata.ColumnType.LOCATION).build()) {
      return tabletsMetadata.stream().collect(Collectors.toList());
    }
  }
}
