/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service.snapshot;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.service.snapshot.TableSnapshot;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotLoader {
    private static final Logger logger = LoggerFactory.getLogger(SnapshotLoader.class);
    static final Pattern SNAPSHOT_DIR_PATTERN = Pattern.compile("(?<keyspace>\\w+)/(?<tableName>\\w+)-(?<tableId>[0-9a-f]{32})/snapshots/(?<tag>.+)$");
    private final Collection<Path> dataDirectories;

    public SnapshotLoader() {
        this(DatabaseDescriptor.getAllDataFileLocations());
    }

    public SnapshotLoader(String[] dataDirectories) {
        this(Arrays.stream(dataDirectories).map(x$0 -> File.getPath(x$0, new String[0])).collect(Collectors.toList()));
    }

    public SnapshotLoader(Collection<Path> dataDirs) {
        this.dataDirectories = dataDirs;
    }

    public SnapshotLoader(Directories directories) {
        this(directories.getCFDirectories().stream().map(File::toPath).collect(Collectors.toList()));
    }

    public Set<TableSnapshot> loadSnapshots(String keyspace) {
        int maxDepth = keyspace == null ? 5 : 4;
        HashMap<String, TableSnapshot.Builder> snapshots = new HashMap<String, TableSnapshot.Builder>();
        Visitor visitor = new Visitor(snapshots);
        for (Path dataDir : this.dataDirectories) {
            if (keyspace != null) {
                dataDir = dataDir.resolve(keyspace);
            }
            try {
                if (new File(dataDir).exists()) {
                    Files.walkFileTree(dataDir, Collections.emptySet(), maxDepth, visitor);
                    continue;
                }
                logger.debug("Skipping non-existing data directory {}", (Object)dataDir);
            }
            catch (IOException e) {
                throw new RuntimeException(String.format("Error while loading snapshots from %s", dataDir), e);
            }
        }
        return snapshots.values().stream().map(TableSnapshot.Builder::build).collect(Collectors.toSet());
    }

    public Set<TableSnapshot> loadSnapshots() {
        return this.loadSnapshots(null);
    }

    @VisibleForTesting
    static class Visitor
    extends SimpleFileVisitor<Path> {
        private static final Pattern UUID_PATTERN = Pattern.compile("([0-9a-f]{8})([0-9a-f]{4})([0-9a-f]{4})([0-9a-f]{4})([0-9a-f]+)");
        private final Map<String, TableSnapshot.Builder> snapshots;

        public Visitor(Map<String, TableSnapshot.Builder> snapshots) {
            this.snapshots = snapshots;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            if (exc instanceof NoSuchFileException) {
                return FileVisitResult.CONTINUE;
            }
            throw exc;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path subdir, BasicFileAttributes attrs) {
            if (subdir.getParent() == null || subdir.getParent().getFileName() == null) {
                return FileVisitResult.CONTINUE;
            }
            if (subdir.getParent().getFileName().toString().equals("snapshots")) {
                logger.trace("Processing directory " + subdir);
                Matcher snapshotDirMatcher = SNAPSHOT_DIR_PATTERN.matcher(subdir.toString());
                if (snapshotDirMatcher.find()) {
                    try {
                        this.loadSnapshotFromDir(snapshotDirMatcher, subdir);
                    }
                    catch (Throwable e) {
                        logger.warn("Could not load snapshot from {}.", (Object)subdir, (Object)e);
                    }
                }
                return FileVisitResult.SKIP_SUBTREE;
            }
            return subdir.getFileName().toString().equals("backups") ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
        }

        static UUID parseUUID(String uuidWithoutDashes) throws IllegalArgumentException {
            assert (uuidWithoutDashes.length() == 32 && !uuidWithoutDashes.contains("-"));
            String dashedUUID = UUID_PATTERN.matcher(uuidWithoutDashes).replaceFirst("$1-$2-$3-$4-$5");
            return UUID.fromString(dashedUUID);
        }

        private void loadSnapshotFromDir(Matcher snapshotDirMatcher, Path snapshotDir) {
            String keyspaceName = snapshotDirMatcher.group("keyspace");
            String tableName = snapshotDirMatcher.group("tableName");
            UUID tableId = Visitor.parseUUID(snapshotDirMatcher.group("tableId"));
            String tag = snapshotDirMatcher.group("tag");
            String snapshotId = TableSnapshot.buildSnapshotId(keyspaceName, tableName, tableId, tag);
            TableSnapshot.Builder builder = this.snapshots.computeIfAbsent(snapshotId, k -> new TableSnapshot.Builder(keyspaceName, tableName, tableId, tag));
            builder.addSnapshotDir(new File(snapshotDir));
        }
    }
}

