/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.indexing.common.actions;

import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.druid.guice.ManageLifecycle;
import org.apache.druid.indexing.common.LockGranularity;
import org.apache.druid.indexing.common.actions.SegmentAllocateAction;
import org.apache.druid.indexing.common.actions.SegmentAllocateRequest;
import org.apache.druid.indexing.common.actions.SegmentAllocateResult;
import org.apache.druid.indexing.common.task.IndexTaskUtils;
import org.apache.druid.indexing.common.task.Task;
import org.apache.druid.indexing.overlord.GlobalTaskLockbox;
import org.apache.druid.indexing.overlord.IndexerMetadataStorageCoordinator;
import org.apache.druid.indexing.overlord.config.TaskLockConfig;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.Intervals;
import org.apache.druid.java.util.common.concurrent.ScheduledExecutorFactory;
import org.apache.druid.java.util.common.granularity.Granularity;
import org.apache.druid.java.util.common.guava.Comparators;
import org.apache.druid.java.util.common.lifecycle.LifecycleStart;
import org.apache.druid.java.util.common.lifecycle.LifecycleStop;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.java.util.emitter.service.ServiceEmitter;
import org.apache.druid.java.util.emitter.service.ServiceEventBuilder;
import org.apache.druid.java.util.emitter.service.ServiceMetricEvent;
import org.apache.druid.segment.realtime.appenderator.SegmentIdWithShardSpec;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.Partitions;
import org.joda.time.Interval;
import org.joda.time.ReadableInterval;

@ManageLifecycle
public class SegmentAllocationQueue {
    private static final Logger log = new Logger(SegmentAllocationQueue.class);
    private static final int MAX_QUEUE_SIZE = 2000;
    private static final int MAX_BATCH_SIZE = 500;
    private final long maxWaitTimeMillis;
    private final GlobalTaskLockbox taskLockbox;
    private final IndexerMetadataStorageCoordinator metadataStorage;
    private final AtomicBoolean isLeader = new AtomicBoolean(false);
    private final ServiceEmitter emitter;
    private final ScheduledExecutorService managerExec;
    private final ScheduledExecutorService workerExec;
    private final Set<String> runningDatasources = Collections.synchronizedSet(new HashSet());
    private final AtomicBoolean isProcessingScheduled = new AtomicBoolean(false);
    private final ConcurrentHashMap<AllocateRequestKey, AllocateRequestBatch> keyToBatch = new ConcurrentHashMap();
    private final BlockingDeque<AllocateRequestBatch> processingQueue = new LinkedBlockingDeque<AllocateRequestBatch>(2000);
    private final TaskLockConfig config;
    private final boolean reduceMetadataIO;

    @Inject
    public SegmentAllocationQueue(GlobalTaskLockbox taskLockbox, TaskLockConfig taskLockConfig, IndexerMetadataStorageCoordinator metadataStorage, ServiceEmitter emitter, ScheduledExecutorFactory executorFactory) {
        this.emitter = emitter;
        this.taskLockbox = taskLockbox;
        this.metadataStorage = metadataStorage;
        this.maxWaitTimeMillis = taskLockConfig.getBatchAllocationWaitTime();
        this.reduceMetadataIO = taskLockConfig.isBatchAllocationReduceMetadataIO();
        this.config = taskLockConfig;
        if (taskLockConfig.isBatchSegmentAllocation()) {
            this.managerExec = executorFactory.create(1, "SegmentAllocQueue-Manager-%s");
            this.workerExec = executorFactory.create(taskLockConfig.getBatchAllocationNumThreads(), "SegmentAllocQueue-Worker-%s");
        } else {
            this.managerExec = null;
            this.workerExec = null;
        }
    }

    @LifecycleStart
    public void start() {
        if (this.isEnabled()) {
            log.info("Initializing segment allocation queue with [%d] worker threads.", new Object[]{this.config.getBatchAllocationNumThreads()});
            this.scheduleQueuePoll(this.maxWaitTimeMillis);
        }
    }

    @LifecycleStop
    public void stop() {
        if (this.isEnabled()) {
            log.info("Tearing down segment allocation queue.", new Object[0]);
            this.managerExec.shutdownNow();
            this.workerExec.shutdownNow();
        }
    }

    public void becomeLeader() {
        if (!this.isLeader.compareAndSet(false, true)) {
            log.info("Already the leader. Queue processing has started.", new Object[0]);
        } else if (this.isEnabled()) {
            log.info("Elected leader. Starting queue processing.", new Object[0]);
        } else {
            log.info("Elected leader but batched segment allocation is disabled. Segment allocation queue will not be used.", new Object[0]);
        }
    }

    public void stopBeingLeader() {
        if (!this.isLeader.compareAndSet(true, false)) {
            log.info("Already surrendered leadership. Queue processing is stopped.", new Object[0]);
        } else if (this.isEnabled()) {
            log.info("Not leader anymore. Stopping queue processing.", new Object[0]);
        } else {
            log.info("Not leader anymore. Segment allocation queue is already disabled.", new Object[0]);
        }
    }

    public boolean isEnabled() {
        return this.managerExec != null && !this.managerExec.isShutdown();
    }

    private void scheduleQueuePoll(long delay) {
        if (this.isProcessingScheduled.compareAndSet(false, true)) {
            this.managerExec.schedule(this::processBatchesDue, delay, TimeUnit.MILLISECONDS);
        }
    }

    public int size() {
        return this.processingQueue.size();
    }

    public Future<SegmentIdWithShardSpec> add(SegmentAllocateRequest request) {
        if (!this.isLeader.get()) {
            throw new ISE("Cannot allocate segment if not leader.", new Object[0]);
        }
        if (!this.isEnabled()) {
            throw new ISE("Batched segment allocation is disabled.", new Object[0]);
        }
        AllocateRequestKey requestKey = new AllocateRequestKey(request);
        AtomicReference futureReference = new AtomicReference();
        this.keyToBatch.compute(requestKey, (key, existingBatch) -> {
            if (existingBatch == null || existingBatch.isStarted() || existingBatch.isFull()) {
                AllocateRequestBatch newBatch = new AllocateRequestBatch((AllocateRequestKey)key);
                futureReference.set(newBatch.add(request));
                return this.addBatchToQueue(newBatch) ? newBatch : null;
            }
            futureReference.set(existingBatch.add(request));
            return existingBatch;
        });
        return (Future)futureReference.get();
    }

    private boolean addBatchToQueue(AllocateRequestBatch batch) {
        batch.resetQueueTime();
        if (!this.isLeader.get()) {
            batch.failPendingRequests("Not leader anymore");
            return false;
        }
        if (this.processingQueue.offer(batch)) {
            log.debug("Added a new batch for key[%s] to queue.", new Object[]{batch.key});
            this.scheduleQueuePoll(this.maxWaitTimeMillis);
            return true;
        }
        batch.failPendingRequests("Segment allocation queue is full. Check the metric `task/action/batch/runTime` to determine if metadata operations are slow.");
        return false;
    }

    private void requeueBatch(AllocateRequestBatch batch) {
        log.info("Requeueing [%d] failed requests in batch [%s].", new Object[]{batch.size(), batch.key});
        this.keyToBatch.compute(batch.key, (key, existingBatch) -> {
            if (existingBatch == null || existingBatch.isFull() || existingBatch.isStarted()) {
                return this.addBatchToQueue(batch) ? batch : null;
            }
            existingBatch.transferRequestsFrom(batch);
            return existingBatch;
        });
    }

    private void processBatchesDue() {
        AllocateRequestBatch nextBatch;
        this.clearQueueIfNotLeader();
        this.isProcessingScheduled.set(false);
        int numSubmittedBatches = 0;
        int numSkippedBatches = 0;
        Iterator<AllocateRequestBatch> queueIterator = this.processingQueue.iterator();
        while (queueIterator.hasNext() && this.runningDatasources.size() < this.config.getBatchAllocationNumThreads()) {
            nextBatch = queueIterator.next();
            String dataSource = nextBatch.key.dataSource;
            if (!nextBatch.isDue()) break;
            if (this.runningDatasources.contains(dataSource)) {
                this.markBatchAsSkipped(nextBatch);
                ++numSkippedBatches;
                continue;
            }
            queueIterator.remove();
            this.runningDatasources.add(dataSource);
            this.workerExec.submit(() -> this.runBatchOnWorker(nextBatch));
            this.emitBatchMetric("task/action/batch/submitted", 1L, nextBatch.key);
            ++numSubmittedBatches;
        }
        log.debug("Submitted [%d] batches, skipped [%d] batches.", new Object[]{numSubmittedBatches, numSkippedBatches});
        if (this.runningDatasources.size() >= this.config.getBatchAllocationNumThreads()) {
            log.debug("Not scheduling again since all worker threads are busy.", new Object[0]);
        } else if (numSkippedBatches >= this.processingQueue.size() || this.processingQueue.isEmpty()) {
            log.debug("Not scheduling again since there are no eligible batches (skipped [%d]).", new Object[]{numSkippedBatches});
        } else {
            nextBatch = this.processingQueue.peek();
            long timeElapsed = System.currentTimeMillis() - nextBatch.getQueueTime();
            long nextScheduleDelay = Math.max(0L, this.maxWaitTimeMillis - timeElapsed);
            this.scheduleQueuePoll(nextScheduleDelay);
            log.debug("Next execution in [%d ms]", new Object[]{nextScheduleDelay});
        }
    }

    private void clearQueueIfNotLeader() {
        int failedBatches = 0;
        AllocateRequestBatch nextBatch = (AllocateRequestBatch)this.processingQueue.peekFirst();
        while (nextBatch != null && !this.isLeader.get()) {
            this.processingQueue.pollFirst();
            this.keyToBatch.remove(nextBatch.key);
            nextBatch.failPendingRequests("Not leader anymore");
            ++failedBatches;
            nextBatch = (AllocateRequestBatch)this.processingQueue.peekFirst();
        }
        if (failedBatches > 0) {
            log.info("Not leader. Failed [%d] batches, remaining in queue [%d].", new Object[]{failedBatches, this.processingQueue.size()});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void runBatchOnWorker(AllocateRequestBatch batch) {
        boolean processed;
        try {
            processed = this.processBatch(batch);
        }
        catch (Throwable t) {
            batch.failPendingRequests(t);
            processed = true;
            log.error(t, "Error while processing batch[%s].", new Object[]{batch.key});
        }
        finally {
            this.runningDatasources.remove(batch.key.dataSource);
        }
        if (!processed) {
            this.requeueBatch(batch);
        }
        this.scheduleQueuePoll(0L);
    }

    private void markBatchAsSkipped(AllocateRequestBatch batch) {
        this.keyToBatch.compute(batch.key, (batchKey, latestBatchForKey) -> {
            if (latestBatchForKey != null) {
                latestBatchForKey.markSkipped();
            }
            return latestBatchForKey;
        });
    }

    private boolean processBatch(AllocateRequestBatch requestBatch) {
        this.keyToBatch.compute(requestBatch.key, (batchKey, latestBatchForKey) -> {
            requestBatch.markStarted();
            return requestBatch.equals(latestBatchForKey) ? null : latestBatchForKey;
        });
        AllocateRequestKey requestKey = requestBatch.key;
        if (requestBatch.isEmpty()) {
            return true;
        }
        if (!this.isLeader.get()) {
            requestBatch.failPendingRequests("Not leader anymore");
            return true;
        }
        log.debug("Processing [%d] requests for batch[%s], queue time[%s].", new Object[]{requestBatch.size(), requestKey, requestBatch.getQueueTime()});
        long startTimeMillis = System.currentTimeMillis();
        int batchSize = requestBatch.size();
        this.emitBatchMetric("task/action/batch/size", batchSize, requestKey);
        this.emitBatchMetric("task/action/batch/queueTime", startTimeMillis - requestBatch.getQueueTime(), requestKey);
        Set<DataSegment> usedSegments = this.retrieveUsedSegments(requestKey);
        int successCount = this.allocateSegmentsForBatch(requestBatch, usedSegments);
        this.emitBatchMetric("task/action/batch/attempts", 1L, requestKey);
        this.emitBatchMetric("task/action/batch/runTime", System.currentTimeMillis() - startTimeMillis, requestKey);
        log.debug("Successfully processed [%d / %d] requests in batch[%s].", new Object[]{successCount, batchSize, requestKey});
        if (requestBatch.isEmpty()) {
            return true;
        }
        log.debug("There are [%d] failed requests in batch[%s].", new Object[]{requestBatch.size(), requestKey});
        Set<DataSegment> updatedUsedSegments = this.retrieveUsedSegments(requestKey);
        if (updatedUsedSegments.equals(usedSegments)) {
            log.warn("Completing [%d] failed requests in batch[%s] with null value as there are conflicting segments. Cannot retry allocation until the set of used segments overlapping the allocation interval[%s] changes.", new Object[]{this.size(), requestKey, requestKey.preferredAllocationInterval});
            requestBatch.completePendingRequestsWithNull();
            return true;
        }
        log.debug("Used segments have changed. Requeuing failed requests.", new Object[0]);
        return false;
    }

    private Set<DataSegment> retrieveUsedSegments(AllocateRequestKey key) {
        return this.metadataStorage.getSegmentTimelineForAllocation(key.dataSource, key.preferredAllocationInterval, key.lockGranularity == LockGranularity.TIME_CHUNK && this.reduceMetadataIO).findNonOvershadowedObjectsInInterval(Intervals.ETERNITY, Partitions.ONLY_COMPLETE);
    }

    private int allocateSegmentsForBatch(AllocateRequestBatch requestBatch, Set<DataSegment> usedSegments) {
        int successCount = 0;
        Set<SegmentAllocateRequest> allRequests = requestBatch.getRequests();
        HashSet<SegmentAllocateRequest> requestsWithNoOverlappingSegment = new HashSet<SegmentAllocateRequest>();
        ArrayList<SegmentAllocateRequest> requestsWithPartialOverlap = new ArrayList<SegmentAllocateRequest>();
        if (usedSegments.isEmpty()) {
            requestsWithNoOverlappingSegment.addAll(allRequests);
        } else {
            Interval[] sortedUsedSegmentIntervals = this.getSortedIntervals(usedSegments);
            HashMap<Interval, List> overlapIntervalToRequests = new HashMap<Interval, List>();
            for (SegmentAllocateRequest segmentAllocateRequest : allRequests) {
                Interval overlappingInterval = Intervals.findOverlappingInterval((Interval)segmentAllocateRequest.getRowInterval(), (Interval[])sortedUsedSegmentIntervals);
                if (overlappingInterval == null) {
                    requestsWithNoOverlappingSegment.add(segmentAllocateRequest);
                    continue;
                }
                if (overlappingInterval.contains((ReadableInterval)segmentAllocateRequest.getRowInterval())) {
                    overlapIntervalToRequests.computeIfAbsent(overlappingInterval, i -> new ArrayList()).add(segmentAllocateRequest);
                    continue;
                }
                requestsWithPartialOverlap.add(segmentAllocateRequest);
            }
            for (Map.Entry entry : overlapIntervalToRequests.entrySet()) {
                successCount += this.allocateSegmentsForInterval((Interval)entry.getKey(), (List)entry.getValue(), requestBatch);
            }
        }
        HashSet<SegmentAllocateRequest> pendingRequests = new HashSet<SegmentAllocateRequest>(requestsWithNoOverlappingSegment);
        List candidateGranularities = Granularity.granularitiesFinerThan((Granularity)requestBatch.key.preferredSegmentGranularity);
        for (Granularity granularity : candidateGranularities) {
            Map<Interval, List<SegmentAllocateRequest>> requestsByInterval = this.getRequestsByInterval(pendingRequests, granularity);
            for (Map.Entry<Interval, List<SegmentAllocateRequest>> entry : requestsByInterval.entrySet()) {
                successCount += this.allocateSegmentsForInterval(entry.getKey(), entry.getValue(), requestBatch);
                pendingRequests.retainAll(requestBatch.getRequests());
            }
        }
        if (!requestsWithPartialOverlap.isEmpty()) {
            log.warn("Found [%d] requests in batch[%s] with row intervals that partially overlap existing segments. These cannot be processed until the set of used segments changes. Example request[%s]", new Object[]{requestsWithPartialOverlap.size(), requestBatch.key, requestsWithPartialOverlap.get(0)});
        }
        return successCount;
    }

    private Interval[] getSortedIntervals(Set<DataSegment> usedSegments) {
        TreeSet sortedSet = new TreeSet(Comparators.intervalsByStartThenEnd());
        usedSegments.forEach(segment -> sortedSet.add(segment.getInterval()));
        return sortedSet.toArray(new Interval[0]);
    }

    private int allocateSegmentsForInterval(Interval tryInterval, List<SegmentAllocateRequest> requests, AllocateRequestBatch requestBatch) {
        if (requests.isEmpty()) {
            return 0;
        }
        AllocateRequestKey requestKey = requestBatch.key;
        log.debug("Trying allocation for [%d] requests, interval[%s] in batch[%s]", new Object[]{requests.size(), tryInterval, requestKey});
        List<SegmentAllocateResult> results = this.taskLockbox.allocateSegments(requests, requestKey.dataSource, tryInterval, requestKey.skipSegmentLineageCheck, requestKey.lockGranularity, this.reduceMetadataIO);
        int successfulRequests = 0;
        for (int i = 0; i < requests.size(); ++i) {
            SegmentAllocateRequest request = requests.get(i);
            SegmentAllocateResult result = results.get(i);
            if (result.isSuccess()) {
                ++successfulRequests;
            }
            requestBatch.handleResult(result, request);
        }
        return successfulRequests;
    }

    private Map<Interval, List<SegmentAllocateRequest>> getRequestsByInterval(Set<SegmentAllocateRequest> requests, Granularity tryGranularity) {
        HashMap<Interval, List<SegmentAllocateRequest>> tryIntervalToRequests = new HashMap<Interval, List<SegmentAllocateRequest>>();
        for (SegmentAllocateRequest request : requests) {
            Interval tryInterval = tryGranularity.bucket(request.getAction().getTimestamp());
            if (!tryInterval.contains((ReadableInterval)request.getRowInterval())) continue;
            tryIntervalToRequests.computeIfAbsent(tryInterval, i -> new ArrayList()).add(request);
        }
        return tryIntervalToRequests;
    }

    private void emitTaskMetric(String metric, long value, SegmentAllocateRequest request) {
        ServiceMetricEvent.Builder metricBuilder = ServiceMetricEvent.builder();
        IndexTaskUtils.setTaskDimensions(metricBuilder, request.getTask());
        metricBuilder.setDimension("taskActionType", (Object)"segmentAllocate");
        this.emitter.emit((ServiceEventBuilder)metricBuilder.setMetric(metric, (Number)value));
    }

    private void emitBatchMetric(String metric, long value, AllocateRequestKey key) {
        ServiceMetricEvent.Builder metricBuilder = ServiceMetricEvent.builder();
        metricBuilder.setDimension("taskActionType", (Object)"segmentAllocate");
        metricBuilder.setDimension("dataSource", (Object)key.dataSource);
        metricBuilder.setDimension("interval", (Object)key.preferredAllocationInterval.toString());
        this.emitter.emit((ServiceEventBuilder)metricBuilder.setMetric(metric, (Number)value));
    }

    private static class AllocateRequestKey {
        private final String dataSource;
        private final String groupId;
        private final Interval preferredAllocationInterval;
        private final Granularity preferredSegmentGranularity;
        private final boolean skipSegmentLineageCheck;
        private final LockGranularity lockGranularity;
        private final boolean useNonRootGenPartitionSpace;
        private final int hash;
        private final String serialized;

        AllocateRequestKey(SegmentAllocateRequest request) {
            SegmentAllocateAction action = request.getAction();
            Task task = request.getTask();
            this.dataSource = action.getDataSource();
            this.groupId = task.getGroupId();
            this.skipSegmentLineageCheck = action.isSkipSegmentLineageCheck();
            this.lockGranularity = action.getLockGranularity();
            this.useNonRootGenPartitionSpace = action.getPartialShardSpec().useNonRootGenerationPartitionSpace();
            this.preferredSegmentGranularity = action.getPreferredSegmentGranularity();
            this.preferredAllocationInterval = action.getPreferredSegmentGranularity().bucket(action.getTimestamp());
            this.hash = Objects.hash(new Object[]{this.dataSource, this.groupId, this.skipSegmentLineageCheck, this.useNonRootGenPartitionSpace, this.preferredAllocationInterval, this.lockGranularity});
            this.serialized = this.serialize();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            AllocateRequestKey that = (AllocateRequestKey)o;
            return this.dataSource.equals(that.dataSource) && this.groupId.equals(that.groupId) && this.skipSegmentLineageCheck == that.skipSegmentLineageCheck && this.useNonRootGenPartitionSpace == that.useNonRootGenPartitionSpace && this.preferredAllocationInterval.equals((Object)that.preferredAllocationInterval) && this.lockGranularity == that.lockGranularity;
        }

        public int hashCode() {
            return this.hash;
        }

        public String toString() {
            return this.serialized;
        }

        private String serialize() {
            return "{datasource='" + this.dataSource + "', groupId='" + this.groupId + "', lock=" + String.valueOf((Object)this.lockGranularity) + ", allocInterval=" + String.valueOf(this.preferredAllocationInterval) + ", skipLineageCheck=" + this.skipSegmentLineageCheck + "}";
        }
    }

    private class AllocateRequestBatch {
        private long queueTimeMillis;
        private final AllocateRequestKey key;
        private boolean started = false;
        private boolean skipped = false;
        private final Map<SegmentAllocateRequest, CompletableFuture<SegmentIdWithShardSpec>> requestToFuture = new HashMap<SegmentAllocateRequest, CompletableFuture<SegmentIdWithShardSpec>>();

        AllocateRequestBatch(AllocateRequestKey key) {
            this.key = key;
        }

        long getQueueTime() {
            return this.queueTimeMillis;
        }

        boolean isDue() {
            return System.currentTimeMillis() - this.queueTimeMillis >= SegmentAllocationQueue.this.maxWaitTimeMillis;
        }

        void resetQueueTime() {
            this.queueTimeMillis = System.currentTimeMillis();
            this.started = false;
            this.skipped = false;
        }

        void markStarted() {
            this.started = true;
        }

        boolean isStarted() {
            return this.started;
        }

        void markSkipped() {
            if (!this.skipped) {
                this.skipped = true;
                SegmentAllocationQueue.this.emitBatchMetric("task/action/batch/skipped", 1L, this.key);
            }
        }

        boolean isFull() {
            return this.size() >= 500;
        }

        Future<SegmentIdWithShardSpec> add(SegmentAllocateRequest request) {
            log.debug("Adding request to batch [%s]: %s", new Object[]{this.key, request.getAction()});
            return this.requestToFuture.computeIfAbsent(request, req -> new CompletableFuture());
        }

        void transferRequestsFrom(AllocateRequestBatch batch) {
            this.requestToFuture.putAll(batch.requestToFuture);
            batch.requestToFuture.clear();
        }

        Set<SegmentAllocateRequest> getRequests() {
            return new HashSet<SegmentAllocateRequest>(this.requestToFuture.keySet());
        }

        void failPendingRequests(String reason) {
            this.failPendingRequests((Throwable)new ISE(reason, new Object[0]));
        }

        void failPendingRequests(Throwable cause) {
            if (!this.requestToFuture.isEmpty()) {
                log.warn("Failing [%d] requests in batch[%s], reason[%s].", new Object[]{this.size(), this.key, cause.getMessage()});
                this.requestToFuture.values().forEach(future -> future.completeExceptionally(cause));
                this.requestToFuture.keySet().forEach(request -> SegmentAllocationQueue.this.emitTaskMetric("task/action/failed/count", 1L, (SegmentAllocateRequest)request));
                this.requestToFuture.clear();
            }
        }

        void completePendingRequestsWithNull() {
            if (this.requestToFuture.isEmpty()) {
                return;
            }
            this.requestToFuture.values().forEach(future -> future.complete(null));
            this.requestToFuture.keySet().forEach(request -> SegmentAllocationQueue.this.emitTaskMetric("task/action/failed/count", 1L, (SegmentAllocateRequest)request));
            this.requestToFuture.clear();
        }

        void handleResult(SegmentAllocateResult result, SegmentAllocateRequest request) {
            request.incrementAttempts();
            if (result.isSuccess()) {
                SegmentAllocationQueue.this.emitTaskMetric("task/action/success/count", 1L, request);
                this.requestToFuture.remove(request).complete(result.getSegmentId());
            } else if (request.canRetry()) {
                log.debug("Allocation failed on attempt [%d] due to error[%s]. Can still retry action[%s].", new Object[]{request.getAttempts(), result.getErrorMessage(), request.getAction()});
            } else {
                SegmentAllocationQueue.this.emitTaskMetric("task/action/failed/count", 1L, request);
                log.error("Exhausted max attempts[%d] for allocation with latest error[%s]. Completing action[%s] with a null value.", new Object[]{request.getAttempts(), result.getErrorMessage(), request.getAction()});
                this.requestToFuture.remove(request).complete(null);
            }
        }

        boolean isEmpty() {
            return this.requestToFuture.isEmpty();
        }

        int size() {
            return this.requestToFuture.size();
        }
    }
}

