/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gravitino.job;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.gravitino.Config;
import org.apache.gravitino.Configs;
import org.apache.gravitino.Entity;
import org.apache.gravitino.EntityAlreadyExistsException;
import org.apache.gravitino.EntityStore;
import org.apache.gravitino.NameIdentifier;
import org.apache.gravitino.Namespace;
import org.apache.gravitino.connector.job.JobExecutor;
import org.apache.gravitino.exceptions.InUseException;
import org.apache.gravitino.exceptions.JobTemplateAlreadyExistsException;
import org.apache.gravitino.exceptions.NoSuchEntityException;
import org.apache.gravitino.exceptions.NoSuchJobException;
import org.apache.gravitino.exceptions.NoSuchJobTemplateException;
import org.apache.gravitino.job.JobExecutorFactory;
import org.apache.gravitino.job.JobHandle;
import org.apache.gravitino.job.JobOperationDispatcher;
import org.apache.gravitino.job.JobTemplate;
import org.apache.gravitino.job.ShellJobTemplate;
import org.apache.gravitino.job.SparkJobTemplate;
import org.apache.gravitino.lock.LockType;
import org.apache.gravitino.lock.TreeLockUtils;
import org.apache.gravitino.meta.AuditInfo;
import org.apache.gravitino.meta.BaseMetalake;
import org.apache.gravitino.meta.JobEntity;
import org.apache.gravitino.meta.JobTemplateEntity;
import org.apache.gravitino.metalake.MetalakeManager;
import org.apache.gravitino.storage.IdGenerator;
import org.apache.gravitino.utils.NameIdentifierUtil;
import org.apache.gravitino.utils.NamespaceUtil;
import org.apache.gravitino.utils.PrincipalUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JobManager
implements JobOperationDispatcher {
    private static final Logger LOG = LoggerFactory.getLogger(JobManager.class);
    private static final Pattern PLACEHOLDER_PATTERN = Pattern.compile("\\{\\{([\\w.-]+)\\}\\}");
    private static final String JOB_STAGING_DIR = File.separator + "%s" + File.separator + "%s" + File.separator + "job-%s";
    private static final long JOB_STAGING_DIR_CLEANUP_MIN_TIME_IN_MS = 600000L;
    private static final long JOB_STATUS_PULL_MIN_INTERVAL_IN_MS = 60000L;
    private static final int TIMEOUT_IN_MS = 30000;
    private final EntityStore entityStore;
    private final File stagingDir;
    private final JobExecutor jobExecutor;
    private final IdGenerator idGenerator;
    private final long jobStagingDirKeepTimeInMs;
    private final ScheduledExecutorService cleanUpExecutor;
    private final ScheduledExecutorService statusPullExecutor;

    public JobManager(Config config, EntityStore entityStore, IdGenerator idGenerator) {
        this(config, entityStore, idGenerator, JobExecutorFactory.create(config));
    }

    @VisibleForTesting
    JobManager(Config config, EntityStore entityStore, IdGenerator idGenerator, JobExecutor jobExecutor) {
        this.entityStore = entityStore;
        this.jobExecutor = jobExecutor;
        this.idGenerator = idGenerator;
        String stagingDirPath = (String)config.get(Configs.JOB_STAGING_DIR);
        this.stagingDir = new File(stagingDirPath);
        if (this.stagingDir.exists()) {
            if (!this.stagingDir.isDirectory()) {
                throw new IllegalArgumentException(String.format("Staging directory %s exists but is not a directory", stagingDirPath));
            }
            if (!(this.stagingDir.canExecute() && this.stagingDir.canRead() && this.stagingDir.canWrite())) {
                throw new IllegalArgumentException(String.format("Staging directory %s is not accessible", stagingDirPath));
            }
        } else if (!this.stagingDir.mkdirs()) {
            throw new IllegalArgumentException(String.format("Failed to create staging directory %s", stagingDirPath));
        }
        this.jobStagingDirKeepTimeInMs = (Long)config.get(Configs.JOB_STAGING_DIR_KEEP_TIME_IN_MS);
        if (this.jobStagingDirKeepTimeInMs < 600000L) {
            LOG.warn("The job staging directory keep time is set to {} ms, the number is too small, which will cause frequent cleanup, please set it to a value larger than {} if you're not using it to do the test.", (Object)this.jobStagingDirKeepTimeInMs, (Object)600000L);
        }
        this.cleanUpExecutor = Executors.newSingleThreadScheduledExecutor(runnable -> {
            Thread thread = new Thread(runnable, "job-staging-dir-cleanup");
            thread.setDaemon(true);
            return thread;
        });
        long scheduleInterval = this.jobStagingDirKeepTimeInMs / 10L;
        Preconditions.checkArgument((scheduleInterval != 0L ? 1 : 0) != 0, (String)"The schedule interval for job staging directory cleanup cannot be zero, please set the job staging directory keep time to a value larger than %s ms", (long)600000L);
        this.cleanUpExecutor.scheduleAtFixedRate(this::cleanUpStagingDirs, scheduleInterval, scheduleInterval, TimeUnit.MILLISECONDS);
        long jobStatusPullIntervalInMs = (Long)config.get(Configs.JOB_STATUS_PULL_INTERVAL_IN_MS);
        if (jobStatusPullIntervalInMs < 60000L) {
            LOG.warn("The job status pull interval is set to {} ms, the number is too small, which will cause frequent job status pull from external job executor, please set it to a value larger than {} if you're not using it to do the test.", (Object)jobStatusPullIntervalInMs, (Object)60000L);
        }
        this.statusPullExecutor = Executors.newSingleThreadScheduledExecutor(runnable -> {
            Thread thread = new Thread(runnable, "job-status-pull");
            thread.setDaemon(true);
            return thread;
        });
        this.statusPullExecutor.scheduleAtFixedRate(this::pullAndUpdateJobStatus, jobStatusPullIntervalInMs, jobStatusPullIntervalInMs, TimeUnit.MILLISECONDS);
    }

    @Override
    public List<JobTemplateEntity> listJobTemplates(String metalake) {
        MetalakeManager.checkMetalake(NameIdentifierUtil.ofMetalake(metalake), this.entityStore);
        Namespace jobTemplateNs = NamespaceUtil.ofJobTemplate(metalake);
        return TreeLockUtils.doWithTreeLock(NameIdentifier.of((String[])jobTemplateNs.levels()), LockType.READ, () -> {
            try {
                return this.entityStore.list(jobTemplateNs, JobTemplateEntity.class, Entity.EntityType.JOB_TEMPLATE);
            }
            catch (IOException ioe) {
                throw new RuntimeException(ioe);
            }
        });
    }

    @Override
    public void registerJobTemplate(String metalake, JobTemplateEntity jobTemplateEntity) throws JobTemplateAlreadyExistsException {
        MetalakeManager.checkMetalake(NameIdentifierUtil.ofMetalake(metalake), this.entityStore);
        NameIdentifier jobTemplateIdent = NameIdentifierUtil.ofJobTemplate(metalake, jobTemplateEntity.name());
        TreeLockUtils.doWithTreeLock(jobTemplateIdent, LockType.WRITE, () -> {
            try {
                this.entityStore.put(jobTemplateEntity, false);
                return null;
            }
            catch (EntityAlreadyExistsException e) {
                throw new JobTemplateAlreadyExistsException("Job template with name %s under metalake %s already exists", new Object[]{jobTemplateEntity.name(), metalake});
            }
            catch (IOException ioe) {
                throw new RuntimeException(ioe);
            }
        });
    }

    @Override
    public JobTemplateEntity getJobTemplate(String metalake, String jobTemplateName) throws NoSuchJobTemplateException {
        MetalakeManager.checkMetalake(NameIdentifierUtil.ofMetalake(metalake), this.entityStore);
        NameIdentifier jobTemplateIdent = NameIdentifierUtil.ofJobTemplate(metalake, jobTemplateName);
        return TreeLockUtils.doWithTreeLock(jobTemplateIdent, LockType.READ, () -> {
            try {
                return this.entityStore.get(jobTemplateIdent, Entity.EntityType.JOB_TEMPLATE, JobTemplateEntity.class);
            }
            catch (NoSuchEntityException e) {
                throw new NoSuchJobTemplateException("Job template with name %s under metalake %s does not exist", new Object[]{jobTemplateName, metalake});
            }
            catch (IOException ioe) {
                throw new RuntimeException(ioe);
            }
        });
    }

    @Override
    public boolean deleteJobTemplate(String metalake, String jobTemplateName) throws InUseException {
        List<JobEntity> jobs;
        MetalakeManager.checkMetalake(NameIdentifierUtil.ofMetalake(metalake), this.entityStore);
        try {
            jobs = this.listJobs(metalake, Optional.of(jobTemplateName));
        }
        catch (NoSuchJobTemplateException e) {
            return false;
        }
        boolean hasActiveJobs = jobs.stream().anyMatch(job -> job.status() != JobHandle.Status.CANCELLED && job.status() != JobHandle.Status.SUCCEEDED && job.status() != JobHandle.Status.FAILED);
        if (hasActiveJobs) {
            throw new InUseException("Job template %s under metalake %s has active jobs associated with it", new Object[]{jobTemplateName, metalake});
        }
        String jobTemplateStagingPath = this.stagingDir.getAbsolutePath() + File.separator + metalake + File.separator + jobTemplateName;
        File jobTemplateStagingDir = new File(jobTemplateStagingPath);
        if (jobTemplateStagingDir.exists()) {
            try {
                FileUtils.deleteDirectory((File)jobTemplateStagingDir);
            }
            catch (IOException e) {
                LOG.error("Failed to delete job template staging directory: {}", (Object)jobTemplateStagingPath, (Object)e);
            }
        }
        return TreeLockUtils.doWithTreeLock(NameIdentifier.of((String[])NamespaceUtil.ofJobTemplate(metalake).levels()), LockType.WRITE, () -> {
            try {
                return this.entityStore.delete(NameIdentifierUtil.ofJobTemplate(metalake, jobTemplateName), Entity.EntityType.JOB_TEMPLATE);
            }
            catch (IOException ioe) {
                throw new RuntimeException(ioe);
            }
        });
    }

    @Override
    public List<JobEntity> listJobs(String metalake, Optional<String> jobTemplateName) throws NoSuchJobTemplateException {
        MetalakeManager.checkMetalake(NameIdentifierUtil.ofMetalake(metalake), this.entityStore);
        Namespace jobNs = NamespaceUtil.ofJob(metalake);
        return TreeLockUtils.doWithTreeLock(NameIdentifier.of((String[])jobNs.levels()), LockType.READ, () -> {
            try {
                List jobEntities;
                jobTemplateName.ifPresent(s -> this.getJobTemplate(metalake, (String)s));
                if (jobTemplateName.isPresent()) {
                    NameIdentifier jobTemplateIdent = NameIdentifierUtil.ofJobTemplate(metalake, (String)jobTemplateName.get());
                    String[] elements = (String[])ArrayUtils.add((Object[])jobTemplateIdent.namespace().levels(), (Object)jobTemplateIdent.name());
                    Namespace jobTemplateIdentNs = Namespace.of((String[])elements);
                    jobEntities = TreeLockUtils.doWithTreeLock(jobTemplateIdent, LockType.READ, () -> this.entityStore.list(jobTemplateIdentNs, JobEntity.class, Entity.EntityType.JOB));
                } else {
                    jobEntities = this.entityStore.list(jobNs, JobEntity.class, Entity.EntityType.JOB);
                }
                return jobEntities;
            }
            catch (IOException ioe) {
                throw new RuntimeException(ioe);
            }
        });
    }

    @Override
    public JobEntity getJob(String metalake, String jobId) throws NoSuchJobException {
        MetalakeManager.checkMetalake(NameIdentifierUtil.ofMetalake(metalake), this.entityStore);
        NameIdentifier jobIdent = NameIdentifierUtil.ofJob(metalake, jobId);
        return TreeLockUtils.doWithTreeLock(jobIdent, LockType.READ, () -> {
            try {
                return this.entityStore.get(jobIdent, Entity.EntityType.JOB, JobEntity.class);
            }
            catch (NoSuchEntityException e) {
                throw new NoSuchJobException("Job with ID %s under metalake %s does not exist", new Object[]{jobId, metalake});
            }
            catch (IOException ioe) {
                throw new RuntimeException(ioe);
            }
        });
    }

    @Override
    public JobEntity runJob(String metalake, String jobTemplateName, Map<String, String> jobConf) throws NoSuchJobTemplateException {
        String jobExecutionId;
        MetalakeManager.checkMetalake(NameIdentifierUtil.ofMetalake(metalake), this.entityStore);
        JobTemplateEntity jobTemplateEntity = this.getJobTemplate(metalake, jobTemplateName);
        long jobId = this.idGenerator.nextId();
        String jobStagingPath = this.stagingDir.getAbsolutePath() + String.format(JOB_STAGING_DIR, metalake, jobTemplateName, jobId);
        File jobStagingDir = new File(jobStagingPath);
        if (!jobStagingDir.mkdirs()) {
            throw new RuntimeException(String.format("Failed to create staging directory %s for job %s", jobStagingDir, jobId));
        }
        JobTemplate jobTemplate = JobManager.createRuntimeJobTemplate(jobTemplateEntity, jobConf, jobStagingDir);
        try {
            jobExecutionId = this.jobExecutor.submitJob(jobTemplate);
        }
        catch (Exception e) {
            throw new RuntimeException(String.format("Failed to submit job template %s for execution", jobTemplate), e);
        }
        JobEntity jobEntity = JobEntity.builder().withId(jobId).withJobExecutionId(jobExecutionId).withJobTemplateName(jobTemplateName).withStatus(JobHandle.Status.QUEUED).withNamespace(NamespaceUtil.ofJob(metalake)).withAuditInfo(AuditInfo.builder().withCreator(PrincipalUtils.getCurrentPrincipal().getName()).withCreateTime(Instant.now()).build()).build();
        try {
            this.entityStore.put(jobEntity, false);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to register the job entity " + String.valueOf(jobEntity), e);
        }
        return jobEntity;
    }

    @Override
    public JobEntity cancelJob(String metalake, String jobId) throws NoSuchJobException {
        MetalakeManager.checkMetalake(NameIdentifierUtil.ofMetalake(metalake), this.entityStore);
        JobEntity jobEntity = this.getJob(metalake, jobId);
        if (jobEntity.status() == JobHandle.Status.CANCELLING || jobEntity.status() == JobHandle.Status.CANCELLED || jobEntity.status() == JobHandle.Status.SUCCEEDED || jobEntity.status() == JobHandle.Status.FAILED) {
            return jobEntity;
        }
        try {
            this.jobExecutor.cancelJob(jobEntity.jobExecutionId());
        }
        catch (Exception e) {
            throw new RuntimeException(String.format("Failed to cancel job with ID %s under metalake %s", jobId, metalake), e);
        }
        JobEntity newJobEntity = JobEntity.builder().withId(jobEntity.id()).withJobExecutionId(jobEntity.jobExecutionId()).withJobTemplateName(jobEntity.jobTemplateName()).withStatus(JobHandle.Status.CANCELLING).withNamespace(jobEntity.namespace()).withAuditInfo(AuditInfo.builder().withCreator(jobEntity.auditInfo().creator()).withCreateTime(jobEntity.auditInfo().createTime()).withLastModifier(PrincipalUtils.getCurrentPrincipal().getName()).withLastModifiedTime(Instant.now()).build()).build();
        return TreeLockUtils.doWithTreeLock(NameIdentifierUtil.ofJob(metalake, jobId), LockType.WRITE, () -> {
            try {
                this.entityStore.put(newJobEntity, true);
                return newJobEntity;
            }
            catch (IOException e) {
                throw new RuntimeException(String.format("Failed to update job entity %s to CANCELING status", newJobEntity), e);
            }
        });
    }

    @Override
    public void close() throws IOException {
        this.jobExecutor.close();
        this.statusPullExecutor.shutdownNow();
        this.cleanUpExecutor.shutdownNow();
    }

    @VisibleForTesting
    void pullAndUpdateJobStatus() {
        List<String> metalakes = JobManager.listInUseMetalakes(this.entityStore);
        for (String metalake : metalakes) {
            List<JobEntity> activeJobs = this.listJobs(metalake, Optional.empty()).stream().filter(job -> job.status() == JobHandle.Status.QUEUED || job.status() == JobHandle.Status.STARTED || job.status() == JobHandle.Status.CANCELLING).toList();
            activeJobs.forEach(job -> {
                JobHandle.Status newStatus = this.jobExecutor.getJobStatus(job.jobExecutionId());
                if (newStatus != job.status()) {
                    JobEntity newJobEntity = JobEntity.builder().withId(job.id()).withJobExecutionId(job.jobExecutionId()).withJobTemplateName(job.jobTemplateName()).withStatus(newStatus).withNamespace(job.namespace()).withAuditInfo(AuditInfo.builder().withCreator(job.auditInfo().creator()).withCreateTime(job.auditInfo().createTime()).withLastModifier(PrincipalUtils.getCurrentPrincipal().getName()).withLastModifiedTime(Instant.now()).build()).build();
                    TreeLockUtils.doWithTreeLock(NameIdentifierUtil.ofJob(metalake, job.name()), LockType.WRITE, () -> {
                        try {
                            this.entityStore.put(newJobEntity, true);
                            return null;
                        }
                        catch (IOException e) {
                            throw new RuntimeException(String.format("Failed to update job entity %s to status %s", newJobEntity, newStatus), e);
                        }
                    });
                    LOG.info("Updated the job {} with execution id {} status to {}", new Object[]{job.name(), job.jobExecutionId(), newStatus});
                }
            });
        }
    }

    @VisibleForTesting
    void cleanUpStagingDirs() {
        List<String> metalakes = JobManager.listInUseMetalakes(this.entityStore);
        for (String metalake : metalakes) {
            List<JobEntity> finishedJobs = this.listJobs(metalake, Optional.empty()).stream().filter(job -> job.status() == JobHandle.Status.CANCELLED || job.status() == JobHandle.Status.SUCCEEDED || job.status() == JobHandle.Status.FAILED).filter(job -> job.finishedAt() > 0L && job.finishedAt() + this.jobStagingDirKeepTimeInMs < System.currentTimeMillis()).toList();
            finishedJobs.forEach(job -> {
                try {
                    this.entityStore.delete(NameIdentifierUtil.ofJob(metalake, job.name()), Entity.EntityType.JOB);
                    String jobStagingPath = this.stagingDir.getAbsolutePath() + String.format(JOB_STAGING_DIR, metalake, job.jobTemplateName(), job.id());
                    File jobStagingDir = new File(jobStagingPath);
                    if (jobStagingDir.exists()) {
                        FileUtils.deleteDirectory((File)jobStagingDir);
                        LOG.info("Deleted job staging directory {} for job {}", (Object)jobStagingPath, (Object)job.name());
                    }
                }
                catch (IOException e) {
                    LOG.error("Failed to delete job and staging directory for job {}", (Object)job.name(), (Object)e);
                }
            });
        }
    }

    @VisibleForTesting
    public static JobTemplate createRuntimeJobTemplate(JobTemplateEntity jobTemplateEntity, Map<String, String> jobConf, File stagingDir) {
        String name = jobTemplateEntity.name();
        String comment = jobTemplateEntity.comment();
        JobTemplateEntity.TemplateContent content = jobTemplateEntity.templateContent();
        String executable = JobManager.fetchFileFromUri(content.executable(), stagingDir, 30000);
        List args = content.arguments().stream().map(arg -> JobManager.replacePlaceholder(arg, jobConf)).collect(Collectors.toList());
        Map<String, String> environments = content.environments().entrySet().stream().collect(Collectors.toMap(entry -> JobManager.replacePlaceholder((String)entry.getKey(), jobConf), entry -> JobManager.replacePlaceholder((String)entry.getValue(), jobConf)));
        Map<String, String> customFields = content.customFields().entrySet().stream().collect(Collectors.toMap(entry -> JobManager.replacePlaceholder((String)entry.getKey(), jobConf), entry -> JobManager.replacePlaceholder((String)entry.getValue(), jobConf)));
        if (content.jobType() == JobTemplate.JobType.SHELL) {
            List<String> scripts = JobManager.fetchFilesFromUri(content.scripts(), stagingDir, 30000);
            return ((ShellJobTemplate.Builder)((ShellJobTemplate.Builder)((ShellJobTemplate.Builder)((ShellJobTemplate.Builder)((ShellJobTemplate.Builder)((ShellJobTemplate.Builder)ShellJobTemplate.builder().withName(name)).withComment(comment)).withExecutable(executable)).withArguments(args)).withEnvironments(environments)).withCustomFields(customFields)).withScripts(scripts).build();
        }
        if (content.jobType() == JobTemplate.JobType.SPARK) {
            String className = content.className();
            List<String> jars = JobManager.fetchFilesFromUri(content.jars(), stagingDir, 30000);
            List<String> files = JobManager.fetchFilesFromUri(content.files(), stagingDir, 30000);
            List<String> archives = JobManager.fetchFilesFromUri(content.archives(), stagingDir, 30000);
            Map<String, String> configs = content.configs().entrySet().stream().collect(Collectors.toMap(entry -> JobManager.replacePlaceholder((String)entry.getKey(), jobConf), entry -> JobManager.replacePlaceholder((String)entry.getValue(), jobConf)));
            return ((SparkJobTemplate.Builder)((SparkJobTemplate.Builder)((SparkJobTemplate.Builder)((SparkJobTemplate.Builder)((SparkJobTemplate.Builder)((SparkJobTemplate.Builder)SparkJobTemplate.builder().withName(name)).withComment(comment)).withExecutable(executable)).withArguments(args)).withEnvironments(environments)).withCustomFields(customFields)).withClassName(className).withJars(jars).withFiles(files).withArchives(archives).withConfigs(configs).build();
        }
        throw new IllegalArgumentException("Unsupported job type: " + String.valueOf(content.jobType()));
    }

    @VisibleForTesting
    static String replacePlaceholder(String inputString, Map<String, String> replacements) {
        if (StringUtils.isBlank((CharSequence)inputString)) {
            return inputString;
        }
        StringBuilder result = new StringBuilder();
        Matcher matcher = PLACEHOLDER_PATTERN.matcher(inputString);
        while (matcher.find()) {
            String key = matcher.group(1);
            String replacement = replacements.get(key);
            if (replacement != null) {
                matcher.appendReplacement(result, replacement);
                continue;
            }
            matcher.appendReplacement(result, matcher.group(0));
        }
        matcher.appendTail(result);
        return result.toString();
    }

    @VisibleForTesting
    static List<String> fetchFilesFromUri(List<String> uris, File stagingDir, int timeoutInMs) {
        return uris.stream().map(uri -> JobManager.fetchFileFromUri(uri, stagingDir, timeoutInMs)).collect(Collectors.toList());
    }

    @VisibleForTesting
    static String fetchFileFromUri(String uri, File stagingDir, int timeoutInMs) {
        try {
            URI fileUri = new URI(uri);
            String scheme = Optional.ofNullable(fileUri.getScheme()).orElse("file");
            File destFile = new File(stagingDir, new File(fileUri.getPath()).getName());
            switch (scheme) {
                case "http": 
                case "https": 
                case "ftp": {
                    FileUtils.copyURLToFile((URL)fileUri.toURL(), (File)destFile, (int)timeoutInMs, (int)timeoutInMs);
                    break;
                }
                case "file": {
                    Files.createSymbolicLink(destFile.toPath(), new File(fileUri.getPath()).toPath(), new FileAttribute[0]);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Unsupported scheme: " + scheme);
                }
            }
            return destFile.getAbsolutePath();
        }
        catch (Exception e) {
            throw new RuntimeException(String.format("Failed to fetch file from URI %s", uri), e);
        }
    }

    private static List<String> listInUseMetalakes(EntityStore entityStore) {
        try {
            List metalakes = TreeLockUtils.doWithRootTreeLock(LockType.READ, () -> entityStore.list(Namespace.empty(), BaseMetalake.class, Entity.EntityType.METALAKE));
            return metalakes.stream().filter(m -> (Boolean)m.propertiesMetadata().getOrDefault(m.properties(), "in-use")).map(BaseMetalake::name).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to list in-use metalakes", e);
        }
    }
}

