/*
 * Decompiled with CFR 0.152.
 */
package io.r2dbc.postgresql.codec;

import io.r2dbc.postgresql.util.Assert;
import java.io.Serializable;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.time.DateTimeException;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.Period;
import java.time.chrono.ChronoPeriod;
import java.time.chrono.Chronology;
import java.time.chrono.IsoChronology;
import java.time.temporal.ChronoUnit;
import java.time.temporal.IsoFields;
import java.time.temporal.Temporal;
import java.time.temporal.TemporalAmount;
import java.time.temporal.TemporalQueries;
import java.time.temporal.TemporalUnit;
import java.time.temporal.UnsupportedTemporalTypeException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.StringTokenizer;
import java.util.concurrent.TimeUnit;

public final class Interval
implements ChronoPeriod,
Serializable {
    public static final Interval ZERO = new Interval(0, 0, 0, 0, 0, 0, 0);
    private static final List<TemporalUnit> SUPPORTED_UNITS = Collections.unmodifiableList(Arrays.asList(ChronoUnit.YEARS, ChronoUnit.MONTHS, ChronoUnit.DAYS, ChronoUnit.HOURS, ChronoUnit.SECONDS, ChronoUnit.MICROS));
    private static final long SECONDS_PER_DAY = 86400L;
    private static final int MICROSECONDS_IN_SECOND = 1000000;
    private static final int SECONDS_IN_MINUTE = 60;
    private static final int SECONDS_IN_HOUR = 3600;
    private static final int SECONDS_IN_DAY = 86400;
    private static final int MONTHS_IN_YEAR = 12;
    private final int years;
    private final int months;
    private final int days;
    private final int hours;
    private final int minutes;
    private final int seconds;
    private final int microseconds;

    private Interval(Period period) {
        this(period, Duration.ZERO);
    }

    private Interval(Duration duration) {
        this(Period.ZERO, duration);
    }

    private Interval(Period period, Duration duration) {
        Assert.requireNonNull(period, "period must not be null");
        Assert.requireNonNull(duration, "duration must not be null");
        long remainderSeconds = duration.getSeconds();
        this.years = period.getYears() + period.getMonths() / 12;
        this.months = period.getMonths() % 12;
        this.days = period.getDays() + (int)(remainderSeconds / 86400L);
        this.hours = (int)(remainderSeconds %= 86400L) / 3600;
        this.minutes = (int)(remainderSeconds %= 3600L) / 60;
        this.seconds = (int)remainderSeconds % 60;
        this.microseconds = (int)TimeUnit.NANOSECONDS.toMicros(duration.getNano());
    }

    private Interval(int years, int months, int days, int hours, int minutes, int seconds, int microseconds) {
        this.years = years;
        this.months = months;
        this.days = days;
        this.hours = hours;
        this.minutes = minutes;
        this.seconds = seconds;
        this.microseconds = microseconds;
    }

    public static Interval of(Period period) {
        Assert.requireNonNull(period, "period must not be null");
        if (Period.ZERO.equals(period)) {
            return ZERO;
        }
        return new Interval(period);
    }

    public static Interval of(Duration duration) {
        Assert.requireNonNull(duration, "duration must not be null");
        if (Duration.ZERO.equals(duration)) {
            return ZERO;
        }
        return new Interval(duration);
    }

    public static Interval of(Period period, Duration duration) {
        Assert.requireNonNull(period, "period must not be null");
        Assert.requireNonNull(duration, "duration must not be null");
        if (Period.ZERO.equals(period) && Duration.ZERO.equals(duration)) {
            return ZERO;
        }
        return new Interval(period, duration);
    }

    public static Interval of(int years, int months, int days, int hours, int minutes, double seconds) {
        if ((years | months | days | hours | minutes) == 0 && seconds == 0.0) {
            return ZERO;
        }
        int secondsInMinute = (int)seconds % 60;
        int microseconds = Math.toIntExact(Math.round((seconds - (double)secondsInMinute) * 1000.0 * 1000.0));
        return new Interval(years, months, days, hours, minutes, secondsInMinute, microseconds);
    }

    public static Interval of(int years, int months, int days, int hours, int minutes, int seconds, int microseconds) {
        if ((years | months | days | hours | minutes | seconds | microseconds) == 0) {
            return ZERO;
        }
        return new Interval(years, months, days, hours, minutes, seconds, microseconds);
    }

    public static Interval between(Temporal startInclusive, Temporal endExclusive) {
        Assert.requireNonNull(startInclusive, "startInclusive must not be null");
        Assert.requireNonNull(endExclusive, "endExclusive must not be null");
        LocalDate startDate = startInclusive.query(TemporalQueries.localDate());
        LocalDate endDate = endExclusive.query(TemporalQueries.localDate());
        Period period = Period.ZERO;
        if (startDate != null && endDate != null) {
            period = Period.between(startDate, endDate);
        }
        LocalTime startTime = startInclusive.query(TemporalQueries.localTime());
        LocalTime endTime = endExclusive.query(TemporalQueries.localTime());
        startTime = startTime != null ? startTime : LocalTime.MIDNIGHT;
        endTime = endTime != null ? endTime : LocalTime.MIDNIGHT;
        Duration duration = Duration.between(startTime, endTime);
        return Interval.of(period, duration);
    }

    public static Interval from(TemporalAmount amount) {
        Assert.requireNonNull(amount, "amount must not be null");
        if (amount instanceof Interval) {
            return (Interval)amount;
        }
        if (amount instanceof Period) {
            return Interval.of((Period)amount);
        }
        if (amount instanceof Duration) {
            return Interval.of((Duration)amount);
        }
        if (amount instanceof ChronoPeriod && !IsoChronology.INSTANCE.equals(((ChronoPeriod)amount).getChronology())) {
            throw new DateTimeException("Period requires ISO chronology: " + amount);
        }
        int years = 0;
        int months = 0;
        int days = 0;
        Duration duration = Duration.ZERO;
        for (TemporalUnit unit : amount.getUnits()) {
            long value = amount.get(unit);
            if (value == 0L) continue;
            if (unit.isDurationEstimated()) {
                if (unit == ChronoUnit.DAYS) {
                    days = Math.addExact(days, Math.toIntExact(value));
                    continue;
                }
                if (unit == ChronoUnit.WEEKS) {
                    days = Math.addExact(days, Math.toIntExact(Math.multiplyExact(value, 7L)));
                    continue;
                }
                if (unit == ChronoUnit.MONTHS) {
                    months = Math.addExact(months, Math.toIntExact(value));
                    continue;
                }
                if (unit == IsoFields.QUARTER_YEARS) {
                    months = Math.addExact(months, Math.toIntExact(Math.multiplyExact(value, 3L)));
                    continue;
                }
                if (unit == ChronoUnit.YEARS) {
                    years = Math.addExact(years, Math.toIntExact(value));
                    continue;
                }
                if (unit == ChronoUnit.DECADES) {
                    years = Math.addExact(years, Math.toIntExact(Math.multiplyExact(value, 10L)));
                    continue;
                }
                if (unit == ChronoUnit.CENTURIES) {
                    years = Math.addExact(years, Math.toIntExact(Math.multiplyExact(value, 100L)));
                    continue;
                }
                if (unit == ChronoUnit.MILLENNIA) {
                    years = Math.addExact(years, Math.toIntExact(Math.multiplyExact(value, 1000L)));
                    continue;
                }
                throw new DateTimeException("Unknown unit: " + unit);
            }
            duration = duration.plus(amount.get(unit), unit);
        }
        return Interval.of(Period.of(years, months, days), duration);
    }

    public static Interval parse(String value) {
        Assert.requireNonNull(value, "value must not be null");
        return value.startsWith("P") ? Interval.parseISO8601IntervalValue(value) : Interval.parsePostgresIntervalValue(value);
    }

    public int getYears() {
        return this.years;
    }

    public int getMonths() {
        return this.months;
    }

    public int getDays() {
        return this.days;
    }

    public int getHours() {
        return this.hours;
    }

    public int getMinutes() {
        return this.minutes;
    }

    public double getSeconds() {
        return (double)this.seconds + (double)this.microseconds / 1000000.0;
    }

    public int getSecondsInMinute() {
        return this.seconds;
    }

    public int getMicrosecondsInSecond() {
        return this.microseconds;
    }

    @Override
    public long get(TemporalUnit unit) {
        if (unit instanceof ChronoUnit) {
            switch ((ChronoUnit)unit) {
                case YEARS: {
                    return this.years;
                }
                case MONTHS: {
                    return this.months;
                }
                case DAYS: {
                    return this.days;
                }
                case HOURS: {
                    return this.hours;
                }
                case MINUTES: {
                    return this.minutes;
                }
                case SECONDS: {
                    return this.seconds;
                }
                case MICROS: {
                    return this.microseconds;
                }
            }
        }
        throw new UnsupportedTemporalTypeException(String.format("Unsupported unit: %s", unit));
    }

    @Override
    public List<TemporalUnit> getUnits() {
        return SUPPORTED_UNITS;
    }

    @Override
    public Temporal addTo(Temporal temporal) {
        Assert.requireNonNull(temporal, "temporal must not be null");
        return temporal.plus(this.getPeriod()).plus(this.getDuration());
    }

    @Override
    public Temporal subtractFrom(Temporal temporal) {
        Assert.requireNonNull(temporal, "temporal must not be null");
        return temporal.minus(this.getPeriod()).minus(this.getDuration());
    }

    @Override
    public Interval plus(TemporalAmount amountToAdd) {
        Assert.requireNonNull(amountToAdd, "amountToAdd must not be null");
        Interval other = Interval.from(amountToAdd);
        return Interval.of(this.getPeriod().plus(other.getPeriod()), this.getDuration().plus(other.getDuration()));
    }

    @Override
    public Interval minus(TemporalAmount amountToSubtract) {
        Assert.requireNonNull(amountToSubtract, "amountToSubtract must not be null");
        Interval other = Interval.from(amountToSubtract);
        return Interval.of(this.getPeriod().minus(other.getPeriod()), this.getDuration().minus(other.getDuration()));
    }

    @Override
    public Chronology getChronology() {
        return IsoChronology.INSTANCE;
    }

    @Override
    public Interval multipliedBy(int scalar) {
        return Interval.of(Math.multiplyExact(this.years, scalar), Math.multiplyExact(this.months, scalar), Math.multiplyExact(this.days, scalar), Math.multiplyExact(this.hours, scalar), Math.multiplyExact(this.minutes, scalar), Math.multiplyExact(this.seconds, scalar), Math.multiplyExact(this.microseconds, scalar));
    }

    @Override
    public Interval normalized() {
        long remainderMicroseconds = (long)this.microseconds + TimeUnit.SECONDS.toMicros(this.seconds) + TimeUnit.MINUTES.toMicros(this.minutes) + TimeUnit.HOURS.toMicros(this.hours) + TimeUnit.DAYS.toMicros(this.days);
        int days = Math.toIntExact(remainderMicroseconds / 86400000000L);
        int hours = Math.toIntExact((remainderMicroseconds -= (long)days * 86400000000L) / 3600000000L);
        int minutes = Math.toIntExact((remainderMicroseconds -= (long)hours * 3600000000L) / 60000000L);
        int seconds = Math.toIntExact((remainderMicroseconds -= (long)minutes * 60000000L) / 1000000L);
        int microseconds = Math.toIntExact((remainderMicroseconds -= (long)seconds * 1000000L) % 1000000L);
        int totalMonths = this.months + this.years * 12;
        int splitYears = totalMonths / 12;
        int splitMonths = totalMonths % 12;
        return Interval.of(splitYears, splitMonths, days, hours, minutes, seconds, microseconds);
    }

    public Interval withYears(int years) {
        return new Interval(years, this.months, this.days, this.hours, this.minutes, this.seconds, this.microseconds);
    }

    public Interval withMonths(int months) {
        return new Interval(this.years, months, this.days, this.hours, this.minutes, this.seconds, this.microseconds);
    }

    public Interval withDays(int days) {
        return new Interval(this.years, this.months, days, this.hours, this.minutes, this.seconds, this.microseconds);
    }

    public Interval withHours(int hours) {
        return new Interval(this.years, this.months, this.days, hours, this.minutes, this.seconds, this.microseconds);
    }

    public Interval withMinutes(int minutes) {
        return new Interval(this.years, this.months, this.days, this.hours, minutes, this.seconds, this.microseconds);
    }

    public Interval withSeconds(int seconds) {
        return new Interval(this.years, this.months, this.days, this.hours, this.minutes, seconds, this.microseconds);
    }

    public Interval withMicroseconds(int microseconds) {
        return new Interval(this.years, this.months, this.days, this.hours, this.minutes, this.seconds, microseconds);
    }

    public Period getPeriod() {
        return Period.of(this.years, this.months, this.days);
    }

    public Duration getDuration() {
        return Duration.ofHours(this.hours).plusMinutes(this.minutes).plusSeconds(this.seconds).plus((long)this.microseconds, ChronoUnit.MICROS);
    }

    public String getValue() {
        DecimalFormat df = (DecimalFormat)NumberFormat.getInstance(Locale.US);
        df.applyPattern("0.0#####");
        return String.format(Locale.ROOT, "%d years %d mons %d days %d hours %d mins %s secs", this.years, this.months, this.days, this.hours, this.minutes, df.format(this.getSeconds()));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Interval)) {
            return false;
        }
        Interval interval = (Interval)o;
        return this.years == interval.years && this.months == interval.months && this.days == interval.days && this.hours == interval.hours && this.minutes == interval.minutes && this.seconds == interval.seconds && this.microseconds == interval.microseconds;
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.years, this.months, this.days, this.hours, this.minutes, this.seconds, this.microseconds);
    }

    @Override
    public String toString() {
        return this.getValue();
    }

    private static Interval parseISO8601IntervalValue(String value) {
        int years = 0;
        int months = 0;
        int days = 0;
        int hours = 0;
        int minutes = 0;
        int seconds = 0;
        int microseconds = 0;
        boolean timeFlag = false;
        boolean microsecondFlag = false;
        boolean negativeFlag = false;
        int currentValue = 0;
        int multiplier = 1;
        for (int i = 1; i < value.length(); ++i) {
            char c = value.charAt(i);
            if (c == 'T') {
                timeFlag = true;
                continue;
            }
            if (c == '-') {
                negativeFlag = true;
                continue;
            }
            if (Character.isDigit(c)) {
                currentValue = currentValue * multiplier + (c - 48);
                multiplier = 10;
                continue;
            }
            if (c == 'Y') {
                years = Interval.getValue(currentValue, negativeFlag);
            } else if (!timeFlag && c == 'M') {
                months = Interval.getValue(currentValue, negativeFlag);
            } else if (c == 'D') {
                days = Interval.getValue(currentValue, negativeFlag);
            } else if (c == 'H') {
                hours = Interval.getValue(currentValue, negativeFlag);
            } else if (timeFlag && c == 'M') {
                minutes = Interval.getValue(currentValue, negativeFlag);
            } else if (!(microsecondFlag || c != 'S' && c != '.')) {
                seconds = Interval.getValue(currentValue, negativeFlag);
                microsecondFlag = true;
            } else if (microsecondFlag && c == 'S') {
                microseconds = Interval.getMicroseconds(currentValue, negativeFlag);
            }
            negativeFlag = false;
            currentValue = 0;
            multiplier = 1;
        }
        return Interval.of(years, months, days, hours, minutes, seconds, microseconds);
    }

    private static Interval parsePostgresIntervalValue(String value) {
        boolean isVerbose = value.startsWith("@");
        if (isVerbose && value.length() == 3 && value.charAt(2) == '0') {
            return Interval.of(Duration.ZERO);
        }
        int years = 0;
        int months = 0;
        int days = 0;
        int hours = 0;
        int minutes = 0;
        double seconds = 0.0;
        boolean ago = false;
        try {
            String changedValue = value.replace('+', ' ').replace('@', ' ');
            StringTokenizer st = new StringTokenizer(changedValue);
            while (st.hasMoreTokens()) {
                String token = st.nextToken();
                if (token.equals("ago")) {
                    ago = true;
                    break;
                }
                int endHours = token.indexOf(58);
                if (endHours != -1) {
                    int offset = token.charAt(0) == '-' ? 1 : 0;
                    hours = Integer.parseInt(token.substring(offset, endHours));
                    minutes = Integer.parseInt(token.substring(endHours + 1, endHours + 3));
                    int endMinutes = token.indexOf(58, endHours + 1);
                    if (endMinutes != -1) {
                        NumberFormat numberFormat = NumberFormat.getNumberInstance(Locale.ROOT);
                        seconds = numberFormat.parse(token.substring(endMinutes + 1)).doubleValue();
                    }
                    if (offset == 1) {
                        hours = -hours;
                        minutes = -minutes;
                        seconds = -seconds;
                    }
                    break;
                }
                String valueToken = token;
                if (!st.hasMoreTokens()) {
                    throw new IllegalArgumentException("Invalid interval: " + value);
                }
                token = st.nextToken();
                if (token.startsWith("mill")) {
                    years = Math.addExact(years, Math.toIntExact(Math.multiplyExact(Integer.parseInt(valueToken), 1000)));
                    continue;
                }
                if (token.startsWith("cent") || token.equals("c")) {
                    years = Math.addExact(years, Math.toIntExact(Math.multiplyExact(Integer.parseInt(valueToken), 100)));
                    continue;
                }
                if (token.startsWith("dec")) {
                    years = Math.addExact(years, Math.toIntExact(Math.multiplyExact(Integer.parseInt(valueToken), 10)));
                    continue;
                }
                if (token.startsWith("year") || token.startsWith("yr")) {
                    years = Math.addExact(years, Math.toIntExact(Integer.parseInt(valueToken)));
                    continue;
                }
                if (token.startsWith("mon")) {
                    months = Integer.parseInt(valueToken);
                    continue;
                }
                if (token.startsWith("week") || token.equals("w")) {
                    days = Math.addExact(days, Math.toIntExact(Math.multiplyExact(Integer.parseInt(valueToken), 7)));
                    continue;
                }
                if (token.startsWith("day")) {
                    days = Math.addExact(days, Math.toIntExact(Integer.parseInt(valueToken)));
                    continue;
                }
                if (token.startsWith("hour") || token.startsWith("hr")) {
                    hours = Integer.parseInt(valueToken);
                    continue;
                }
                if (token.startsWith("min")) {
                    minutes = Integer.parseInt(valueToken);
                    continue;
                }
                if (!token.startsWith("sec")) continue;
                seconds = Double.parseDouble(valueToken);
            }
        }
        catch (ParseException e) {
            throw new IllegalArgumentException("Invalid interval: " + value, e);
        }
        if (isVerbose && ago) {
            years = -years;
            months = -months;
            days = -days;
            hours = -hours;
            minutes = -minutes;
            seconds = -seconds;
        }
        return Interval.of(years, months, days, hours, minutes, seconds);
    }

    private static int getValue(int value, boolean negativeFlag) {
        return negativeFlag ? -value : value;
    }

    private static int getMicroseconds(int value, boolean negativeValue) {
        while (value != 0 && value * 10 < 1000000) {
            value *= 10;
        }
        return negativeValue ? -value : value;
    }
}

