/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * license agreements; and to You under the Apache License, version 2.0:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * This file is part of the Apache Pekko project, which was derived from Akka.
 */

/*
 * Copyright (C) 2020-2022 Lightbend Inc. <https://www.lightbend.com>
 */

package jdocs.stream.operators.flow;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletionStage;
import org.apache.pekko.NotUsed;
import org.apache.pekko.actor.ActorSystem;
import org.apache.pekko.japi.Pair;
import org.apache.pekko.stream.javadsl.Flow;
import org.apache.pekko.stream.javadsl.RunnableGraph;
import org.apache.pekko.stream.javadsl.Sink;
import org.apache.pekko.stream.javadsl.Source;

public class Lazy {
  private ActorSystem system = null;

  void example() {
    // #simple-example
    Source<Integer, NotUsed> numbers =
        Source.unfold(
                0,
                n -> {
                  int next = n + 1;
                  System.out.println("Source producing " + next);
                  return Optional.of(Pair.create(next, next));
                })
            .take(3);

    Flow<Integer, Integer, CompletionStage<NotUsed>> flow =
        Flow.lazyFlow(
            () -> {
              System.out.println("Creating the actual flow");
              return Flow.fromFunction(
                  element -> {
                    System.out.println("Actual flow mapped " + element);
                    return element;
                  });
            });

    numbers.via(flow).run(system);
    // prints:
    // Source producing 1
    // Creating the actual flow
    // Actual flow mapped 1
    // Source producing 2
    // Actual flow mapped 2
    // #simple-example
  }

  void statefulMap() {
    // #mutable-example
    Flow<Integer, List<Integer>, CompletionStage<NotUsed>> mutableFold =
        Flow.lazyFlow(
            () -> {
              List<Integer> zero = new ArrayList<>();

              return Flow.of(Integer.class)
                  .fold(
                      zero,
                      (list, element) -> {
                        list.add(element);
                        return list;
                      });
            });

    RunnableGraph<NotUsed> stream =
        Source.range(1, 3).via(mutableFold).to(Sink.foreach(System.out::println));

    stream.run(system);
    stream.run(system);
    stream.run(system);
    // prints:
    // [1, 2, 3]
    // [1, 2, 3]
    // [1, 2, 3]
    // #mutable-example
  }
}
