/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.shuffle;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.spark.network.buffer.ManagedBuffer;
import org.apache.spark.network.sasl.SaslTimeoutException;
import org.apache.spark.network.shuffle.BlockFetchingListener;
import org.apache.spark.network.shuffle.BlockPushingListener;
import org.apache.spark.network.shuffle.BlockTransferListener;
import org.apache.spark.network.shuffle.ErrorHandler;
import org.apache.spark.network.util.NettyUtils;
import org.apache.spark.network.util.TransportConf;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sparkproject.guava.annotations.VisibleForTesting;
import org.sparkproject.guava.base.Preconditions;
import org.sparkproject.guava.collect.Sets;
import org.sparkproject.guava.util.concurrent.Uninterruptibles;

public class RetryingBlockTransferor {
    private static final ExecutorService executorService = Executors.newCachedThreadPool(NettyUtils.createThreadFactory((String)"Block Transfer Retry"));
    private static final Logger logger = LoggerFactory.getLogger(RetryingBlockTransferor.class);
    private final BlockTransferStarter transferStarter;
    private final BlockTransferListener listener;
    private final int maxRetries;
    private final int retryWaitTime;
    private int retryCount = 0;
    private int saslRetryCount = 0;
    private final LinkedHashSet<String> outstandingBlocksIds;
    private RetryingBlockTransferListener currentListener;
    private final boolean enableSaslRetries;
    private final ErrorHandler errorHandler;

    public RetryingBlockTransferor(TransportConf conf, BlockTransferStarter transferStarter, String[] blockIds, BlockTransferListener listener, ErrorHandler errorHandler) {
        this.transferStarter = transferStarter;
        this.listener = listener;
        this.maxRetries = conf.maxIORetries();
        this.retryWaitTime = conf.ioRetryWaitTimeMs();
        this.outstandingBlocksIds = Sets.newLinkedHashSet();
        Collections.addAll(this.outstandingBlocksIds, blockIds);
        this.currentListener = new RetryingBlockTransferListener();
        this.errorHandler = errorHandler;
        this.enableSaslRetries = conf.enableSaslRetries();
        this.saslRetryCount = 0;
    }

    public RetryingBlockTransferor(TransportConf conf, BlockTransferStarter transferStarter, String[] blockIds, BlockFetchingListener listener) {
        this(conf, transferStarter, blockIds, listener, ErrorHandler.NOOP_ERROR_HANDLER);
    }

    public void start() {
        this.transferAllOutstanding();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void transferAllOutstanding() {
        RetryingBlockTransferListener myListener;
        int numRetries;
        String[] blockIdsToTransfer;
        RetryingBlockTransferor retryingBlockTransferor = this;
        synchronized (retryingBlockTransferor) {
            blockIdsToTransfer = this.outstandingBlocksIds.toArray(new String[this.outstandingBlocksIds.size()]);
            numRetries = this.retryCount;
            myListener = this.currentListener;
        }
        try {
            this.transferStarter.createAndStart(blockIdsToTransfer, myListener);
        }
        catch (Exception e) {
            logger.error(String.format("Exception while beginning %s of %s outstanding blocks %s", this.listener.getTransferType(), blockIdsToTransfer.length, numRetries > 0 ? "(after " + numRetries + " retries)" : ""), (Throwable)e);
            if (this.shouldRetry(e)) {
                this.initiateRetry(e);
            }
            for (String bid : blockIdsToTransfer) {
                this.listener.onBlockTransferFailure(bid, e);
            }
        }
    }

    private synchronized void initiateRetry(Throwable e) {
        if (this.enableSaslRetries && e instanceof SaslTimeoutException) {
            ++this.saslRetryCount;
        }
        ++this.retryCount;
        this.currentListener = new RetryingBlockTransferListener();
        logger.info("Retrying {} ({}/{}) for {} outstanding blocks after {} ms", new Object[]{this.listener.getTransferType(), this.retryCount, this.maxRetries, this.outstandingBlocksIds.size(), this.retryWaitTime});
        executorService.submit(() -> {
            Uninterruptibles.sleepUninterruptibly((long)this.retryWaitTime, (TimeUnit)TimeUnit.MILLISECONDS);
            this.transferAllOutstanding();
        });
    }

    private synchronized boolean shouldRetry(Throwable e) {
        boolean isSaslTimeout;
        boolean isIOException = e instanceof IOException || e.getCause() instanceof IOException;
        boolean bl = isSaslTimeout = this.enableSaslRetries && e instanceof SaslTimeoutException;
        if (!isSaslTimeout && this.saslRetryCount > 0) {
            Preconditions.checkState((this.retryCount >= this.saslRetryCount ? 1 : 0) != 0, (Object)"retryCount must be greater than or equal to saslRetryCount");
            this.retryCount -= this.saslRetryCount;
            this.saslRetryCount = 0;
        }
        boolean hasRemainingRetries = this.retryCount < this.maxRetries;
        boolean shouldRetry = (isSaslTimeout || isIOException) && hasRemainingRetries && this.errorHandler.shouldRetryError(e);
        return shouldRetry;
    }

    @VisibleForTesting
    public int getRetryCount() {
        return this.retryCount;
    }

    private class RetryingBlockTransferListener
    implements BlockFetchingListener,
    BlockPushingListener {
        private RetryingBlockTransferListener() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void handleBlockTransferSuccess(String blockId, ManagedBuffer data) {
            boolean shouldForwardSuccess = false;
            RetryingBlockTransferor retryingBlockTransferor = RetryingBlockTransferor.this;
            synchronized (retryingBlockTransferor) {
                if (this == RetryingBlockTransferor.this.currentListener && RetryingBlockTransferor.this.outstandingBlocksIds.contains(blockId)) {
                    RetryingBlockTransferor.this.outstandingBlocksIds.remove(blockId);
                    shouldForwardSuccess = true;
                    if (RetryingBlockTransferor.this.saslRetryCount > 0) {
                        Preconditions.checkState((RetryingBlockTransferor.this.retryCount >= RetryingBlockTransferor.this.saslRetryCount ? 1 : 0) != 0, (Object)"retryCount must be greater than or equal to saslRetryCount");
                        RetryingBlockTransferor.this.retryCount = RetryingBlockTransferor.this.retryCount - RetryingBlockTransferor.this.saslRetryCount;
                        RetryingBlockTransferor.this.saslRetryCount = 0;
                    }
                }
            }
            if (shouldForwardSuccess) {
                RetryingBlockTransferor.this.listener.onBlockTransferSuccess(blockId, data);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void handleBlockTransferFailure(String blockId, Throwable exception) {
            boolean shouldForwardFailure = false;
            RetryingBlockTransferor retryingBlockTransferor = RetryingBlockTransferor.this;
            synchronized (retryingBlockTransferor) {
                if (this == RetryingBlockTransferor.this.currentListener && RetryingBlockTransferor.this.outstandingBlocksIds.contains(blockId)) {
                    if (RetryingBlockTransferor.this.shouldRetry(exception)) {
                        RetryingBlockTransferor.this.initiateRetry(exception);
                    } else {
                        if (RetryingBlockTransferor.this.errorHandler.shouldLogError(exception)) {
                            logger.error(String.format("Failed to %s block %s, and will not retry (%s retries)", RetryingBlockTransferor.this.listener.getTransferType(), blockId, RetryingBlockTransferor.this.retryCount), exception);
                        } else {
                            logger.debug(String.format("Failed to %s block %s, and will not retry (%s retries)", RetryingBlockTransferor.this.listener.getTransferType(), blockId, RetryingBlockTransferor.this.retryCount), exception);
                        }
                        RetryingBlockTransferor.this.outstandingBlocksIds.remove(blockId);
                        shouldForwardFailure = true;
                    }
                }
            }
            if (shouldForwardFailure) {
                RetryingBlockTransferor.this.listener.onBlockTransferFailure(blockId, exception);
            }
        }

        @Override
        public void onBlockFetchSuccess(String blockId, ManagedBuffer data) {
            this.handleBlockTransferSuccess(blockId, data);
        }

        @Override
        public void onBlockFetchFailure(String blockId, Throwable exception) {
            this.handleBlockTransferFailure(blockId, exception);
        }

        @Override
        public void onBlockPushSuccess(String blockId, ManagedBuffer data) {
            this.handleBlockTransferSuccess(blockId, data);
        }

        @Override
        public void onBlockPushFailure(String blockId, Throwable exception) {
            this.handleBlockTransferFailure(blockId, exception);
        }

        @Override
        public void onBlockTransferSuccess(String blockId, ManagedBuffer data) {
            throw new RuntimeException("Invocation on RetryingBlockTransferListener.onBlockTransferSuccess is unexpected.");
        }

        @Override
        public void onBlockTransferFailure(String blockId, Throwable exception) {
            throw new RuntimeException("Invocation on RetryingBlockTransferListener.onBlockTransferFailure is unexpected.");
        }

        @Override
        public String getTransferType() {
            throw new RuntimeException("Invocation on RetryingBlockTransferListener.getTransferType is unexpected.");
        }
    }

    public static interface BlockTransferStarter {
        public void createAndStart(String[] var1, BlockTransferListener var2) throws IOException, InterruptedException;
    }
}

